<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';

/* =====================
   Allow POST only
===================== */
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: index.php');
    exit;
}

session_start();

$order_token = $_POST['order_token'] ?? '';

if (
    empty($order_token) ||
    empty($_SESSION['order_token']) ||
    !hash_equals($_SESSION['order_token'], $order_token)
) {
    die('Duplicate or invalid submission detected.');
}

/* =====================
   1. Validate input
===================== */
$employee_id    = trim($_POST['employee_id'] ?? '');
$employee_name  = trim($_POST['employee_name'] ?? '');
$employee_email = trim($_POST['employee_email'] ?? '');
$employee_phone = trim($_POST['employee_phone'] ?? '');
$order_for_date = $_POST['order_for_date'] ?? '';
$qtys           = $_POST['qty'] ?? [];
$morning_pickup_time = $_POST['morning_pickup_time'] ?? null;

if ($morning_pickup_time === '') {
    $morning_pickup_time = null;
}

if (!$employee_id || !$employee_name || !$employee_email || !$order_for_date) {
    die("Missing required fields.");
}

unset($_SESSION['order_token']);

/* =====================
   2. Build order items
===================== */
$total = 0;
$order_items = [];

foreach ($qtys as $menu_id => $qty) {
    $qty = (int)$qty;
    if ($qty > 0) {
        $stmt = $pdo->prepare("
            SELECT id, title, price, meal_session
            FROM menu_items
            WHERE id = ? AND published = 1
        ");
        $stmt->execute([$menu_id]);
        $item = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($item) {
            $order_items[] = [
                'menu_id'      => $item['id'],
                'title'        => $item['title'],
                'qty'          => $qty,
                'price'        => $item['price'],
                'meal_session' => $item['meal_session']
            ];
            $total += $item['price'] * $qty;
        }
    }
}

if (empty($order_items)) {
    die("No items selected.");
}

/* =====================
   Morning pickup validation
===================== */
$hasMorning = false;
foreach ($order_items as $it) {
    if ($it['meal_session'] === 'morning') {
        $hasMorning = true;
        break;
    }
}

if ($hasMorning && !$morning_pickup_time) {
    die("Please select a morning pickup time.");
}

/* =====================
   3. Generate order code
===================== */
$short_date = date("ymd", strtotime($order_for_date));

$stmt = $pdo->prepare("
    SELECT order_code
    FROM orders
    WHERE order_for_date = ?
    ORDER BY id DESC
    LIMIT 1
");
$stmt->execute([$order_for_date]);
$last = $stmt->fetch(PDO::FETCH_ASSOC);

$seq = $last ? intval(explode('-', $last['order_code'])[1]) + 1 : 1;
$order_code = $short_date . '-' . str_pad($seq, 3, '0', STR_PAD_LEFT);

/* =====================
   4. Insert order + items (TRANSACTION)
===================== */
$pdo->beginTransaction();

try {

    /* Insert order */
    $stmt = $pdo->prepare("
        INSERT INTO orders
        (
            order_code,
            employee_id,
            employee_name,
            employee_email,
            employee_phone,
            order_for_date,
            morning_pickup_time,
            total_amount
        )
        VALUES (?, ?, ?, ?, ?, ?, ?, ?)
    ");

    $stmt->execute([
        $order_code,
        $employee_id,
        $employee_name,
        $employee_email,
        $employee_phone,
        $order_for_date,
        $morning_pickup_time,
        $total
    ]);

    $order_id = $pdo->lastInsertId();

    /* Insert items */
    foreach ($order_items as $it) {
        $stmt = $pdo->prepare("
            INSERT INTO order_items (order_id, menu_item_id, qty, price)
            VALUES (?, ?, ?, ?)
        ");
        $stmt->execute([
            $order_id,
            $it['menu_id'],
            $it['qty'],
            $it['price']
        ]);
    }

    $pdo->commit();

} catch (Exception $e) {
    $pdo->rollBack();
    die("Order failed: " . $e->getMessage());
}

/* =====================
   5. Group items
===================== */
$grouped = [
    'morning'   => [],
    'afternoon' => []
];

foreach ($order_items as $it) {
    $grouped[$it['meal_session']][] = $it;
}

/* =====================
   6. Email to EMPLOYEE
===================== */
$employee_subject = "Canteen Order Received - {$order_code}";

$employee_body = "
<h3>Order Received</h3>
<p>
<strong>Order ID:</strong> {$order_code}<br>
<strong>Food Date:</strong> {$order_for_date}<br>";

if ($morning_pickup_time) {
    $employee_body .= "<strong>Morning Pickup:</strong> "
        . date('g:i A', strtotime($morning_pickup_time)) . "<br>";
}

$employee_body .= "
<strong>Total:</strong> RM " . number_format($total, 2) . "
</p>
";

if (!empty($grouped['morning'])) {
    $employee_body .= "<h4>Morning Order</h4><ul>";
    foreach ($grouped['morning'] as $it) {
        $employee_body .= "<li>{$it['title']} × {$it['qty']}</li>";
    }
    $employee_body .= "</ul>";
}

if (!empty($grouped['afternoon'])) {
    $employee_body .= "<h4>Afternoon Order</h4><ul>";
    foreach ($grouped['afternoon'] as $it) {
        $employee_body .= "<li>{$it['title']} × {$it['qty']}</li>";
    }
    $employee_body .= "</ul>";
}

$employee_body .= "
<p>
Please send your <strong>Order ID ({$order_code})</strong> to the canteen admin via WhatsApp.
</p>
<p>Thank you,<br>" . SITE_NAME . "</p>
";

try {
    send_email($employee_email, $employee_subject, $employee_body);
} catch (Throwable $e) {
    error_log($e->getMessage());
}

/* =====================
   7. Email to ADMIN
===================== */
$admin_subject = "New Canteen Order - {$order_code}";

$admin_body = "
<h3>New Canteen Order</h3>
<p>
<strong>Order ID:</strong> {$order_code}<br>
<strong>Employee:</strong> {$employee_name} ({$employee_id})<br>
<strong>Food Date:</strong> {$order_for_date}<br>";

if ($morning_pickup_time) {
    $admin_body .= "<strong>Morning Pickup:</strong> "
        . date('g:i A', strtotime($morning_pickup_time)) . "<br>";
}

$admin_body .= "
<strong>Total:</strong> RM " . number_format($total, 2) . "
</p>
";

foreach ($grouped as $session => $list) {
    if ($list) {
        $admin_body .= "<h4>" . ucfirst($session) . " Items</h4><ul>";
        foreach ($list as $it) {
            $admin_body .= "<li>{$it['title']} × {$it['qty']}</li>";
        }
        $admin_body .= "</ul>";
    }
}

foreach (ADMIN_EMAILS as $adminEmail) {
    try {
        send_email($adminEmail, $admin_subject, $admin_body);
    } catch (Throwable $e) {
        error_log($e->getMessage());
    }
}

/* =====================
   8. Success page
===================== */
header("Location: order_success.php?code=" . urlencode($order_code));
exit;
