<?php
ini_set('display_errors', 1);
error_reporting(E_ALL);

date_default_timezone_set('Asia/Kuala_Lumpur');

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../libs/fpdf.php';

/* =========================
   TOMORROW DATE
========================= */
$foodDate = date('Y-m-d', strtotime('+1 day'));

/* =========================
   TOTAL ORDERS
========================= */
$stmt = $pdo->prepare("
    SELECT COUNT(*) 
    FROM orders
    WHERE order_for_date = ?
      AND status = 'approved'
");
$stmt->execute([$foodDate]);
$totalOrders = (int)$stmt->fetchColumn();

/* =========================
   TOTAL QTY
========================= */
$stmt = $pdo->prepare("
    SELECT IFNULL(SUM(oi.qty), 0)
    FROM orders o
    JOIN order_items oi ON oi.order_id = o.id
    WHERE o.order_for_date = ?
      AND o.status = 'approved'
");
$stmt->execute([$foodDate]);
$totalQty = (int)$stmt->fetchColumn();

/* =========================
   TOTAL SALES
========================= */
$stmt = $pdo->prepare("
    SELECT IFNULL(SUM(oi.qty * oi.price), 0)
    FROM orders o
    JOIN order_items oi ON oi.order_id = o.id
    WHERE o.order_for_date = ?
      AND o.status = 'approved'
");
$stmt->execute([$foodDate]);
$totalAmount = (float)$stmt->fetchColumn();

/* =========================
   STOP IF NO ORDER
========================= */
if ($totalOrders == 0) {
    exit;
}

/* =========================
   PDF FUNCTIONS
========================= */
function pdf_text($text) {
    return iconv('UTF-8', 'ISO-8859-1//IGNORE', $text);
}

class PDF extends FPDF {
    function Header() {
        $this->SetFont('Arial','B',16);
        $this->Cell(0,10,'CANTEEN SALES SUMMARY',0,1,'C');
        $this->Ln(4);
    }
}

/* =========================
   GET ITEM SUMMARY
========================= */
$stmt = $pdo->prepare("
    SELECT
        mi.title AS item_name,
        SUM(oi.qty) AS total_qty,
        SUM(oi.qty * oi.price) AS total_amount
    FROM orders o
    JOIN order_items oi ON oi.order_id = o.id
    JOIN menu_items mi ON mi.id = oi.menu_item_id
    WHERE o.order_for_date = ?
      AND o.status = 'approved'
    GROUP BY oi.menu_item_id
    ORDER BY mi.title
");
$stmt->execute([$foodDate]);
$items = $stmt->fetchAll(PDO::FETCH_ASSOC);

/* =========================
   CREATE PDF
========================= */
$pdf = new PDF();
$pdf->AddPage();

$pdf->SetFont('Arial','',12);
$pdf->Cell(
    0,
    8,
    pdf_text("Food Date: " . date('d M Y', strtotime($foodDate))),
    0,
    1
);
$pdf->Ln(4);

$pdf->SetFont('Arial','B',12);
$pdf->Cell(100,9,"Item",1);
$pdf->Cell(30,9,"Qty",1,0,'C');
$pdf->Cell(60,9,"Amount (RM)",1,1,'R');

$pdf->SetFont('Arial','',11);

foreach ($items as $row) {
    $pdf->Cell(100,8,pdf_text($row['item_name']),1);
    $pdf->Cell(30,8,$row['total_qty'],1,0,'C');
    $pdf->Cell(60,8,"RM ".number_format($row['total_amount'],2),1,1,'R');
}

$pdf->SetFont('Arial','B',12);
$pdf->Cell(130,10,"GRAND TOTAL",1);
$pdf->Cell(60,10,"RM ".number_format($totalAmount,2),1,1,'R');

/* =========================
   SAVE PDF TEMP
========================= */
$pdfFile = sys_get_temp_dir() . "/Sales_Report_{$foodDate}.pdf";
$pdf->Output('F', $pdfFile);

/* =========================
   EMAIL BODY
========================= */
$subject = "Canteen Sales Report - " . date('d M Y', strtotime($foodDate));

$body = "
<h3>Canteen Sales Summary</h3>

<p>
<strong>Food Date:</strong> " . date('d M Y', strtotime($foodDate)) . "<br><br>

<strong>Total Orders:</strong> {$totalOrders}<br>
<strong>Total Order Quantity:</strong> {$totalQty}<br>
<strong>Total Sales:</strong> RM " . number_format($totalAmount, 2) . "
</p>

<p>
Please refer to the attached PDF for item breakdown.
</p>

<p>
Regards,<br>
" . SITE_NAME . "
</p>
";

/* =========================
   SEND EMAIL WITH ATTACHMENT
========================= */
foreach (SALES_REPORT_EMAILS as $email) {
    try {
      send_email_with_attachment(
    $email,
    $subject,
    $body,
    [$pdfFile]
);
    } catch (Throwable $e) {
        error_log($e->getMessage());
    }
}

/* =========================
   CLEAN UP
========================= */
if (file_exists($pdfFile)) {
    unlink($pdfFile);
}

echo "Sales report with PDF sent successfully.";

