<?php
session_start();

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';

if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

/* =========================================================
   1️⃣ SINGLE ORDER VIEW (APPROVE / REJECT)
   This MUST come FIRST
========================================================= */


// Preserve filters for links
$filterQuery = http_build_query([
    'order_for_date' => $_GET['order_for_date'] ?? '',
    'status'         => $_GET['status'] ?? '',
    'employee_name'  => $_GET['employee_name'] ?? '',
    'order_code'     => $_GET['order_code'] ?? ''
]);

if (isset($_GET['order_id'])) {

    $id = (int)$_GET['order_id'];

    $stmt = $pdo->prepare("SELECT * FROM orders WHERE id = ?");
    $stmt->execute([$id]);
    $order = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$order) {
        die("Order not found.");
    }

    $stmt_items = $pdo->prepare("
        SELECT 
            oi.*, 
            mi.title,
            mi.meal_session
        FROM order_items oi
        JOIN menu_items mi ON mi.id = oi.menu_item_id
        WHERE oi.order_id = ?
        ORDER BY mi.meal_session, mi.title
    ");
    $stmt_items->execute([$id]);
    $items = $stmt_items->fetchAll(PDO::FETCH_ASSOC);

    include __DIR__ . '/../includes/header.php';
    ?>

    <p>
       <a href="view_orders.php?<?= $filterQuery ?>" class="btn btn-secondary">
← Back to Orders
</a>
    </p>

    <h4>Order <?= htmlspecialchars($order['order_code']) ?></h4>
    <p><strong>Employee:</strong> <?= htmlspecialchars($order['employee_name']) ?> (<?= htmlspecialchars($order['employee_id']) ?>)</p>
    <p><strong>Food Date:</strong> <?= htmlspecialchars($order['order_for_date']) ?></p>
    <?php if (!empty($order['morning_pickup_time'])): ?>
<p>
    <strong>Morning Pickup Time:</strong>
    <?= date('g:i A', strtotime($order['morning_pickup_time'])) ?>
</p>
<?php endif; ?>
    <p><strong>Total:</strong> RM <?= number_format($order['total_amount'], 2) ?></p>

    <hr>

    <?php
    $morning = [];
    $afternoon = [];

    foreach ($items as $it) {
        if ($it['meal_session'] === 'morning') {
            $morning[] = $it;
        } else {
            $afternoon[] = $it;
        }
    }
    ?>

   <?php if ($morning): ?>
<h6 class="mt-3">🌅 Morning Order</h6>
<ul>
<?php foreach ($morning as $it): 
    $subtotal = $it['qty'] * $it['price'];
?>
<li>
<?= htmlspecialchars($it['title']) ?>
— <?= (int)$it['qty'] ?> × RM <?= number_format($it['price'],2) ?>
= <strong>RM <?= number_format($subtotal,2) ?></strong>
</li>
<?php endforeach; ?>
</ul>
<?php endif; ?>


  <?php if ($afternoon): ?>
<h6 class="mt-3">🌞 Afternoon Order</h6>
<ul>
<?php foreach ($afternoon as $it): 
    $subtotal = $it['qty'] * $it['price'];
?>
<li>
<?= htmlspecialchars($it['title']) ?>
— <?= (int)$it['qty'] ?> × RM <?= number_format($it['price'],2) ?>
= <strong>RM <?= number_format($subtotal,2) ?></strong>
</li>
<?php endforeach; ?>
</ul>
<?php endif; ?>


    <hr>

    <?php if ($order['status'] === 'pending'): ?>
        <form method="post" action="approve_order.php">
            <input type="hidden" name="order_id" value="<?= (int)$order['id'] ?>">

            <div class="mb-3">
                <label class="form-label">Admin Note</label>
                <textarea name="admin_note" class="form-control"></textarea>
            </div>

            <div class="mb-3">
                <label class="form-label">Action</label>
                <select name="action" class="form-select">
                    <option value="approve">Approve</option>
                    <option value="reject">Reject</option>
                </select>
            </div>

            <button class="btn btn-primary">Submit</button>
        </form>
    <?php else: ?>
        <div class="alert alert-info">
            Order is <strong><?= htmlspecialchars($order['status']) ?></strong>.
        </div>
    <?php endif; ?>

    <?php
    include __DIR__ . '/../includes/footer.php';
    exit;
}

/* =========================================================
   2️⃣ LIST / FILTER MODE
========================================================= */

$order_date = $_GET['order_for_date'] ?? date('Y-m-d');
$status = strtolower(trim($_GET['status'] ?? ''));
$employee_name = trim($_GET['employee_name'] ?? '');
$order_code = trim($_GET['order_code'] ?? '');

$where = " WHERE 1 ";
$params = [];

if ($order_date) {
    $where .= " AND o.order_for_date = ? ";
    $params[] = $order_date;
}

if ($status !== '') {
    $where .= " AND LOWER(o.status) = ? ";
    $params[] = $status;
}

if ($employee_name !== '') {
    $where .= " AND o.employee_name LIKE ? ";
    $params[] = "%$employee_name%";
}

if ($order_code !== '') {
    $where .= " AND o.order_code LIKE ? ";
    $params[] = "%$order_code%";
}

$sql = "
    SELECT * 
    FROM orders o
    $where
    ORDER BY created_at DESC
    LIMIT 200
";

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$orders = $stmt->fetchAll(PDO::FETCH_ASSOC);

include __DIR__ . '/../includes/header.php';
?>

<p>
    <a href="dashboard.php" class="btn btn-secondary mb-3">
        &larr; Back to Dashboard
    </a>
</p>

<h4>All Orders</h4>

<form method="get" class="row g-3 mb-4">
    <div class="col-md-3">
        <label class="form-label">Employee Name</label>
        <input type="text" name="employee_name" class="form-control"
               value="<?= htmlspecialchars($employee_name) ?>">
    </div>

    <div class="col-md-3">
        <label class="form-label">Order Date</label>
        <input type="date" name="order_for_date" class="form-control"
               value="<?= htmlspecialchars($order_date) ?>">
    </div>

    <div class="col-md-3">
        <label class="form-label">Status</label>
        <select name="status" class="form-select">
            <option value="">-- All --</option>
            <option value="pending" <?= $status === 'pending' ? 'selected' : '' ?>>Pending</option>
            <option value="approved" <?= $status === 'approved' ? 'selected' : '' ?>>Approved</option>
            <option value="rejected" <?= $status === 'rejected' ? 'selected' : '' ?>>Rejected</option>
        </select>
    </div>

    <div class="col-md-3">
        <label class="form-label">Order Code</label>
        <input type="text" name="order_code" class="form-control"
               value="<?= htmlspecialchars($order_code) ?>">
    </div>

    <div class="col-12 text-end">
        <button class="btn btn-primary">Apply Filter</button>
        <a href="view_orders.php" class="btn btn-secondary">Reset</a>
    </div>
</form>

<a href="export_summary.php?<?= http_build_query($_GET) ?>"
   class="btn btn-success mb-3">
   Export PDF Summary
</a>

<?php if ($orders): ?>
<table class="table table-bordered">
    <thead>
        <tr>
            <th>Order</th>
            <th>Employee</th>
            <th>For Date</th>
            <th>Total</th>
            <th>Status</th>
            <th>Time</th>
        </tr>
    </thead>
    <tbody>
        <?php foreach ($orders as $o): ?>
        <tr>
            <td>
              <a href="view_orders.php?order_id=<?= $o['id'] ?>&<?= $filterQuery ?>">
    <?= htmlspecialchars($o['order_code']) ?>
</a>
            </td>
            <td><?= htmlspecialchars($o['employee_name']) ?></td>
            <td><?= htmlspecialchars($o['order_for_date']) ?></td>
            <td>RM <?= number_format($o['total_amount'], 2) ?></td>
            <td><?= htmlspecialchars($o['status']) ?></td>
            <td><?= htmlspecialchars($o['created_at']) ?></td>
        </tr>
        <?php endforeach; ?>
    </tbody>
</table>
<?php else: ?>
<p>No orders found.</p>
<?php endif; ?>

<?php include __DIR__ . '/../includes/footer.php'; ?>
