<?php
session_start();

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../libs/fpdf.php';

if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

/* =======================
   UTF-8 → FPDF SAFE TEXT
======================= */
function pdf_text($text) {
    return iconv('UTF-8', 'ISO-8859-1//TRANSLIT', $text);
}

/* =======================
   READ FILTERS
======================= */
$order_date = $_GET['order_for_date'] ?? '';
if (!$order_date) {
    die("Food date is required.");
}
$status = strtolower(trim($_GET['status'] ?? ''));

/* =======================
   WHERE
======================= */
$where = " WHERE o.order_for_date = ? ";
$params = [$order_date];

if ($status !== '') {
    $where .= " AND LOWER(o.status) = ? ";
    $params[] = $status;
}

/* =======================
   MORNING (SPLIT BY PICKUP)
======================= */
$sql_morning = "
SELECT
    o.morning_pickup_time,
    mi.title,
    SUM(oi.qty) AS total_qty
FROM orders o
JOIN order_items oi ON oi.order_id = o.id
JOIN menu_items mi ON mi.id = oi.menu_item_id
{$where}
AND mi.meal_session = 'morning'
GROUP BY o.morning_pickup_time, mi.id
ORDER BY o.morning_pickup_time, mi.title
";
$stmt = $pdo->prepare($sql_morning);
$stmt->execute($params);
$morningItems = $stmt->fetchAll(PDO::FETCH_ASSOC);

/* =======================
   AFTERNOON SUMMARY
======================= */
$sql_afternoon = "
SELECT
    mi.title,
    SUM(oi.qty) AS total_qty
FROM orders o
JOIN order_items oi ON oi.order_id = o.id
JOIN menu_items mi ON mi.id = oi.menu_item_id
{$where}
AND mi.meal_session = 'afternoon'
GROUP BY mi.id
ORDER BY mi.title
";
$stmt2 = $pdo->prepare($sql_afternoon);
$stmt2->execute($params);
$afternoonItems = $stmt2->fetchAll(PDO::FETCH_ASSOC);

/* =======================
   TRACEABILITY
======================= */
$sql_employee = "
SELECT
    o.order_code,
    o.employee_name,
    o.employee_id,
    o.morning_pickup_time,
    mi.meal_session,
    mi.title,
    oi.qty
FROM orders o
JOIN order_items oi ON oi.order_id = o.id
JOIN menu_items mi ON mi.id = oi.menu_item_id
{$where}
ORDER BY
    o.morning_pickup_time,
    o.employee_name,
    o.order_code,
    mi.meal_session,
    mi.title
";
$stmt3 = $pdo->prepare($sql_employee);
$stmt3->execute($params);
$employee_orders = $stmt3->fetchAll(PDO::FETCH_ASSOC);

/* =======================
   START PDF
======================= */
$pdf = new FPDF();
$pdf->AddPage();

/* ---------- PAGE 1: KITCHEN ---------- */
$pdf->SetFont('Arial','B',16);
$pdf->Cell(0,10,pdf_text("CANTEEN PREPARATION LIST"),0,1,'C');
$pdf->Ln(3);

$pdf->SetFont('Arial','',12);
$pdf->Cell(0,8,pdf_text("Food Date: " . date('d M Y', strtotime($order_date))),0,1);
$pdf->Ln(5);

/* ===== MORNING ===== */
$pdf->SetFont('Arial','B',14);
$pdf->Cell(0,9,pdf_text("MORNING"),0,1);

$currentPickup = '';
$pdf->SetFont('Arial','',12);

foreach ($morningItems as $row) {

    if ($row['morning_pickup_time'] !== $currentPickup) {
        $currentPickup = $row['morning_pickup_time'];
        $label = $currentPickup
            ? date('g:i A', strtotime($currentPickup))
            : 'N/A';

        $pdf->Ln(3);
        $pdf->SetFont('Arial','B',13);
        $pdf->Cell(0,8,pdf_text("Pickup Time: {$label}"),0,1);
        $pdf->SetFont('Arial','',12);
    }

   // Save position
$x = $pdf->GetX();
$y = $pdf->GetY();

// Item title (auto wrap)
$pdf->MultiCell(150, 8, pdf_text($row['title']), 0);
$rowHeight = $pdf->GetY() - $y;

// Qty column (aligned)
$pdf->SetXY($x + 150, $y);
$pdf->Cell(30, $rowHeight, $row['total_qty'], 0, 1, 'C');
}

/* ===== AFTERNOON ===== */
if ($afternoonItems) {
    $pdf->Ln(6);
    $pdf->SetFont('Arial','B',14);
    $pdf->Cell(0,9,pdf_text("AFTERNOON"),0,1);

    $pdf->SetFont('Arial','',12);

    foreach ($afternoonItems as $row) {
      $x = $pdf->GetX();
$y = $pdf->GetY();

$pdf->MultiCell(150, 8, pdf_text($row['title']), 0);
$rowHeight = $pdf->GetY() - $y;

$pdf->SetXY($x + 150, $y);
$pdf->Cell(30, $rowHeight, $row['total_qty'], 0, 1, 'C');

    }
}

/* ---------- PAGE 2: TRACEABILITY ---------- */
$pdf->AddPage();

$pdf->SetFont('Arial','B',16);
$pdf->Cell(0,10,pdf_text("ORDER BREAKDOWN (TRACEABILITY)"),0,1,'C');
$pdf->Ln(5);

/* ===== GROUP DATA ===== */
$grouped = [];
foreach ($employee_orders as $row) {
    $pickup = $row['morning_pickup_time'] ?: 'N/A';
    $key = $pickup.'|'.$row['employee_name'].'|'.$row['order_code'];
    $grouped[$key][] = $row;
}

$lastPickup = '';

foreach ($grouped as $key => $rows) {

    [$pickup, $employee, $orderCode] = explode('|', $key);

    if ($pickup !== $lastPickup) {
        $pdf->Ln(4);
        $pdf->SetFont('Arial','B',13);
        $pdf->Cell(
            0,
            8,
            pdf_text("PICKUP TIME: " . ($pickup !== 'N/A' ? date('g:i A', strtotime($pickup)) : 'N/A')),
            0,
            1
        );
        $lastPickup = $pickup;
    }

    $startY = $pdf->GetY();
    $startX = 10;

    $pdf->SetFont('Arial','B',12);
    $pdf->Cell(
        190,
        10,
        pdf_text("{$employee} ({$orderCode})"),
        1,
        1
    );

    $currentSession = '';

    foreach ($rows as $r) {

        if ($currentSession !== $r['meal_session']) {
            $pdf->SetFont('Arial','I',11);
            $pdf->Cell(190,8,pdf_text(strtoupper($r['meal_session'])),1,1);
            $currentSession = $r['meal_session'];
        }

        $rowY = $pdf->GetY();

        $pdf->SetFont('Arial','',11);

// Save position
$x = $pdf->GetX();
$y = $pdf->GetY();

// Item name (auto wrap)
$pdf->MultiCell(
    160,
    8,
    pdf_text('- '.$r['title']),
    0
);

$rowHeight = $pdf->GetY() - $y;

// Qty column
$pdf->SetXY($x + 160, $y);
$pdf->Cell(30, $rowHeight, $r['qty'], 0, 1, 'C');

// Vertical divider
$pdf->Line($startX + 160, $y, $startX + 160, $y + $rowHeight);
    }

    $endY = $pdf->GetY();
    $pdf->Rect($startX, $startY, 190, $endY - $startY);

    $pdf->Ln(6);
}

$pdf->Output("D", "Canteen_Preparation_{$order_date}.pdf");
exit;
