<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

session_start();

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../libs/fpdf.php';

if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

/* =======================
   UTF-8 SAFE
======================= */
function pdf_text($text) {
    if ($text === null) return '';
    return iconv('UTF-8', 'ISO-8859-1//IGNORE', $text);
}

/* =======================
   CUSTOM PDF CLASS
======================= */
class PDF extends FPDF {

    function Header() {
        $this->SetFont('Arial','B',16);
        $this->Cell(0,10,'CANTEEN SALES SUMMARY',0,1,'C');
        $this->Ln(3);
    }

}

/* =======================
   INPUT
======================= */
$dateFrom = $_GET['date_from'] ?? '';
$dateTo   = $_GET['date_to'] ?? '';

if (!$dateFrom || !$dateTo) {
    die("Date range is required.");
}

/* =======================
   SALES SUMMARY
======================= */
$sql = "
SELECT
    mi.title AS item_name,
    SUM(oi.qty) AS total_qty,
    SUM(oi.qty * oi.price) AS total_amount
FROM orders o
JOIN order_items oi ON oi.order_id = o.id
JOIN menu_items mi ON mi.id = oi.menu_item_id
WHERE o.order_for_date BETWEEN ? AND ?
  AND o.status = 'approved'
GROUP BY oi.menu_item_id
ORDER BY mi.title
";

$stmt = $pdo->prepare($sql);
$stmt->execute([$dateFrom, $dateTo]);
$items = $stmt->fetchAll(PDO::FETCH_ASSOC);

/* =======================
   GRAND TOTAL
======================= */
$grandStmt = $pdo->prepare("
    SELECT IFNULL(SUM(oi.qty * oi.price), 0)
    FROM orders o
    JOIN order_items oi ON oi.order_id = o.id
    WHERE o.order_for_date BETWEEN ? AND ?
      AND o.status = 'approved'
");
$grandStmt->execute([$dateFrom, $dateTo]);
$grandTotal = $grandStmt->fetchColumn();

/* =======================
   PDF OUTPUT
======================= */
$pdf = new PDF();
$pdf->AddPage();

/* Date line */
$pdf->SetFont('Arial','',12);
$pdf->Cell(
    0,
    8,
    pdf_text(
        "Food Date: " .
        date('d M Y', strtotime($dateFrom)) .
        " to " .
        date('d M Y', strtotime($dateTo))
    ),
    0,
    1
);
$pdf->Ln(5);

/* Table header */
$pdf->SetFont('Arial','B',12);
$pdf->Cell(100,9,"Item",1);
$pdf->Cell(30,9,"Total Qty",1,0,'C');
$pdf->Cell(60,9,"Total Amount (RM)",1,1,'R');

/* Table body */
$pdf->SetFont('Arial','',11);

foreach ($items as $row) {

    $itemText = pdf_text($row['item_name']);
    $qty      = (int)$row['total_qty'];
    $amount   = "RM " . number_format($row['total_amount'], 2);

    // 🔒 SAFE maximum height (4 lines x 8 = 32)
    $maxRowHeight = 32;

    // If row + safety margin doesn't fit, new page FIRST
    if ($pdf->GetY() + $maxRowHeight > ($pdf->GetPageHeight() - 10)) {
        $pdf->AddPage();

        // Repeat table header
        $pdf->SetFont('Arial','B',12);
        $pdf->Cell(100,9,"Item",1);
        $pdf->Cell(30,9,"Total Qty",1,0,'C');
        $pdf->Cell(60,9,"Total Amount (RM)",1,1,'R');
        $pdf->SetFont('Arial','',11);
    }

    $x = $pdf->GetX();
    $y = $pdf->GetY();

    // Item
    $pdf->MultiCell(100, 8, $itemText, 1);
    $rowHeight = $pdf->GetY() - $y;

    // Qty
    $pdf->SetXY($x + 100, $y);
    $pdf->Cell(30, $rowHeight, $qty, 1, 0, 'C');

    // Amount
    $pdf->Cell(60, $rowHeight, $amount, 1, 1, 'R');
}


/* Grand total */
$pdf->SetFont('Arial','B',12);
$pdf->Cell(130,10,"GRAND TOTAL",1);
$pdf->Cell(60,10,"RM " . number_format($grandTotal, 2),1,1,'R');

/* Output */
$pdf->Output(
    "D",
    "Sales_Summary_{$dateFrom}_to_{$dateTo}.pdf"
);
exit;
