<?php
session_start();
require_once __DIR__ . '/../includes/db.php';

$id = intval($_GET['id'] ?? 0);
if ($id <= 0) {
    die("Invalid menu ID.");
}

/* ---------- Fetch menu item ---------- */
$stmt = $pdo->prepare("SELECT * FROM menu_items WHERE id = ?");
$stmt->execute([$id]);
$item = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$item) {
    die("Menu item not found.");
}

$uploadDir = __DIR__ . '/../uploads/';
if (!is_dir($uploadDir)) mkdir($uploadDir, 0755, true);

$error = '';
$success = '';

/* ---------- Handle update ---------- */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    $title = trim($_POST['title']);
    $description = trim($_POST['description'] ?? '');
    $price = floatval($_POST['price'] ?? 0);
    $meal_session = $_POST['meal_session'] ?? 'morning';
    $published = isset($_POST['published']) ? 1 : 0;
    $imagePath = $item['image'];

    // Image upload (optional)
    if (isset($_FILES['image']) && $_FILES['image']['error'] === 0) {
        $ext = pathinfo($_FILES['image']['name'], PATHINFO_EXTENSION);
        $newName = uniqid('menu_') . "." . $ext;

        if (move_uploaded_file($_FILES['image']['tmp_name'], $uploadDir . $newName)) {
            $imagePath = $newName;
        } else {
            $error = "Image upload failed!";
        }
    }

    if ($title === '') {
        $error = "Title cannot be empty!";
    } elseif ($price <= 0) {
        $error = "Price must be greater than 0!";
    }

    if (empty($error)) {
        $stmt = $pdo->prepare("
            UPDATE menu_items
            SET title = ?, description = ?, price = ?, meal_session = ?, published = ?, image = ?
            WHERE id = ?
        ");

        if ($stmt->execute([
            $title,
            $description,
            $price,
            $meal_session,
            $published,
            $imagePath,
            $id
        ])) {

            $success = "Menu updated successfully!";

            // 🔄 Refresh item data
            $stmt = $pdo->prepare("SELECT * FROM menu_items WHERE id = ?");
            $stmt->execute([$id]);
            $item = $stmt->fetch(PDO::FETCH_ASSOC);

        } else {
            $error = "Failed to update menu.";
        }
    }
}
?>

<!DOCTYPE html>
<html>
<head>
    <title>Edit Menu</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body class="bg-light">

<div class="container py-5">
    <h2>Edit Menu Item</h2>

    <a href="add_menu.php" class="btn btn-secondary mb-3">&larr; Back</a>

    <?php if ($error): ?>
        <div class="alert alert-danger"><?= htmlspecialchars($error) ?></div>
    <?php endif; ?>

    <?php if ($success): ?>
        <div class="alert alert-success"><?= htmlspecialchars($success) ?></div>
    <?php endif; ?>

    <form method="post" enctype="multipart/form-data">

        <div class="mb-3">
            <label class="form-label">Menu Title</label>
            <input type="text" class="form-control" name="title"
                   value="<?= htmlspecialchars($item['title']) ?>" required>
        </div>

        <div class="mb-3">
            <label class="form-label">Description</label>
            <textarea class="form-control" name="description"><?= htmlspecialchars($item['description']) ?></textarea>
        </div>

        <div class="mb-3">
            <label class="form-label">Price (RM)</label>
            <input type="number" class="form-control" name="price"
                   value="<?= htmlspecialchars($item['price']) ?>" step="0.01" required>
        </div>

        <!-- ✅ Meal Session -->
        <div class="mb-3">
            <label class="form-label">Meal Session</label>
            <select name="meal_session" class="form-select" required>
                <option value="morning" <?= $item['meal_session'] === 'morning' ? 'selected' : '' ?>>
                    Morning
                </option>
                <option value="afternoon" <?= $item['meal_session'] === 'afternoon' ? 'selected' : '' ?>>
                    Afternoon
                </option>
            </select>
        </div>

        <div class="mb-3">
            <label class="form-label">Current Image</label><br>
            <?php if ($item['image']): ?>
                <img src="../uploads/<?= htmlspecialchars($item['image']) ?>"
                     width="120" class="rounded mb-2">
            <?php else: ?>
                <p class="text-muted">No image</p>
            <?php endif; ?>
        </div>

        <div class="mb-3">
            <label class="form-label">Upload New Image (optional)</label>
            <input type="file" class="form-control" name="image" accept="image/*">
        </div>

        <div class="form-check mb-3">
            <input type="checkbox" class="form-check-input" name="published" value="1"
                   <?= $item['published'] ? 'checked' : '' ?>>
            <label class="form-check-label">Published</label>
        </div>

        <button class="btn btn-primary" type="submit">Save Changes</button>

    </form>
</div>

</body>
</html>
