<?php
session_start();

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';

if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

$page_title = "Admin Dashboard";
include __DIR__ . '/../includes/header.php';

/* ======================
   Order date control
====================== */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['active_order_date'])) {
    $stmt = $pdo->prepare("
        UPDATE order_settings
        SET active_order_date = ?
        WHERE id = 1
    ");
    $stmt->execute([$_POST['active_order_date']]);
}

$activeDate = $pdo->query("
    SELECT active_order_date
    FROM order_settings
    WHERE id = 1
")->fetchColumn();

/* ======================
   Stats
====================== */
$stats = $pdo->prepare("
    SELECT
        COUNT(*) AS total,
        SUM(status = 'pending') AS pending,
        SUM(status = 'approved') AS approved
    FROM orders
    WHERE order_for_date = ?
");
$stats->execute([$activeDate]);
$stat = $stats->fetch(PDO::FETCH_ASSOC);

/* ======================
   Orders
====================== */
$stmt = $pdo->prepare("
    SELECT *
    FROM orders
    WHERE order_for_date = ?
    ORDER BY created_at DESC
");
$stmt->execute([$activeDate]);
$orders = $stmt->fetchAll(PDO::FETCH_ASSOC);

/* ======================
   Approved Grand Total
====================== */
$grandStmt = $pdo->prepare("
    SELECT IFNULL(SUM(total_amount), 0)
    FROM orders
    WHERE order_for_date = ?
      AND status = 'approved'
");
$grandStmt->execute([$activeDate]);
$approvedGrandTotal = $grandStmt->fetchColumn();
?>

<style>
.dashboard-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    border-bottom: 1px solid #e5e7eb;
    padding-bottom: 12px;
    margin-bottom: 20px;
}
.stat-bar {
    display: flex;
    gap: 40px;
    padding: 12px 0;
    border-bottom: 1px solid #e5e7eb;
    margin-bottom: 20px;
}
.stat-item span {
    display: block;
    font-size: 13px;
    color: #6b7280;
}
.stat-item strong {
    font-size: 20px;
    font-weight: 600;
}
.table thead th {
    font-size: 13px;
    text-transform: uppercase;
    color: #6b7280;
}
.badge {
    font-weight: 500;
}
</style>

<!-- ================= HEADER ================= -->
<div class="dashboard-header">
    <div>
        <h4 class="mb-1">Canteen Orders</h4>
        <small class="text-muted">
            Logged in as <?= htmlspecialchars($_SESSION['admin_name']) ?>
        </small>
    </div>

    <form method="post" id="dateForm" class="d-flex align-items-center gap-2">
        <label class="small text-muted">Food Date</label>
        <input type="date"
               id="dateInput"
               name="active_order_date"
               value="<?= htmlspecialchars($activeDate) ?>"
               data-original="<?= htmlspecialchars($activeDate) ?>"
               class="form-control form-control-sm"
               required>
        <button class="btn btn-sm btn-outline-primary">
            Apply
        </button>
    </form>
</div>

<!-- ================= STATS ================= -->
<div class="stat-bar">
    <div class="stat-item">
        <span>Total Orders</span>
        <strong><?= (int)$stat['total'] ?></strong>
    </div>
    <div class="stat-item">
        <span>Pending</span>
        <strong class="text-warning"><?= (int)$stat['pending'] ?></strong>
    </div>
    <div class="stat-item">
        <span>Approved</span>
        <strong class="text-success"><?= (int)$stat['approved'] ?></strong>
    </div>
    <div class="stat-item">
        <span>Approved Total</span>
        <strong class="text-success">
            RM <?= number_format($approvedGrandTotal, 2) ?>
        </strong>
    </div>
</div>

<!-- ================= ACTIONS ================= -->
<div class="mb-3 d-flex gap-2 flex-wrap">
    <a href="view_orders.php?order_date=<?= $activeDate ?>"
       class="btn btn-sm btn-outline-secondary">
        View All Orders
    </a>

    <a href="add_menu.php"
       class="btn btn-sm btn-outline-secondary">
        Add Menu Item
    </a>
</div>

<!-- ================= EXPORT SALES ================= -->
<!-- ================= EXPORT SALES (DATE RANGE) ================= -->
<div class="mb-4">
    <form action="export_sales_report.php"
          method="get"
          target="_blank"
          class="d-flex align-items-end gap-2 flex-wrap">

        <div>
            <label class="form-label small text-muted mb-1">
                From Date
            </label>
            <input type="date"
                   name="date_from"
                   value="<?= htmlspecialchars($activeDate) ?>"
                   class="form-control form-control-sm"
                   required>
        </div>

        <div>
            <label class="form-label small text-muted mb-1">
                To Date
            </label>
            <input type="date"
                   name="date_to"
                   value="<?= htmlspecialchars($activeDate) ?>"
                   class="form-control form-control-sm"
                   required>
        </div>

        <button type="submit"
                class="btn btn-sm btn-outline-success">
            📊 Export Sales Summary (PDF)
        </button>
    </form>
</div>


<!-- ================= TABLE ================= -->
<div class="table-responsive">
<table class="table table-hover">
    <thead>
        <tr>
            <th>Order ID</th>
            <th>Employee</th>
            <th>Total</th>
            <th>Status</th>
            <th>Food Date</th>
            <th>Created</th>
        </tr>
    </thead>
    <tbody>
    <?php if ($orders): ?>
        <?php foreach ($orders as $o): ?>
        <tr>
            <td>
                <a href="view_orders.php?order_id=<?= $o['id'] ?>">
                    <?= htmlspecialchars($o['order_code']) ?>
                </a>
            </td>
            <td><?= htmlspecialchars($o['employee_name']) ?></td>
            <td>RM <?= number_format($o['total_amount'], 2) ?></td>
            <td>
                <span class="badge bg-<?=
                    $o['status'] === 'pending' ? 'warning' :
                    ($o['status'] === 'approved' ? 'success' : 'secondary')
                ?>">
                    <?= ucfirst($o['status']) ?>
                </span>
            </td>
            <td><?= date('d M Y', strtotime($o['order_for_date'])) ?></td>
            <td><?= date('d M Y, g:i A', strtotime($o['created_at'])) ?></td>
        </tr>
        <?php endforeach; ?>
    <?php else: ?>
        <tr>
            <td colspan="6" class="text-center text-muted">
                No orders for this date
            </td>
        </tr>
    <?php endif; ?>
    </tbody>
</table>
</div>

<script>
document.getElementById('dateForm').addEventListener('submit', function(e) {
    const input = document.getElementById('dateInput');
    if (input.value !== input.dataset.original) {
        const confirmation = prompt(
            'To proceed with changing the active order date to ' +
            input.value +
            ', please type "CONFIRM"'
        );
        if (confirmation !== 'CONFIRM') {
            e.preventDefault();
        }
    }
});
</script>

<?php include __DIR__ . '/../includes/footer.php'; ?>
