<?php
session_start();

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';

if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: view_orders.php');
    exit;
}

/* =========================
   READ INPUT
========================= */
$order_id   = (int)($_POST['order_id'] ?? 0);
$action     = $_POST['action'] ?? '';
$admin_note = trim($_POST['admin_note'] ?? '');

if (!$order_id || !in_array($action, ['approve', 'reject'], true)) {
    die("Invalid request.");
}

/* =========================
   FETCH ORDER
========================= */
$stmt = $pdo->prepare("SELECT * FROM orders WHERE id = ?");
$stmt->execute([$order_id]);
$order = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$order) {
    die("Order not found.");
}

/* =========================
   UPDATE ORDER STATUS
========================= */
$new_status = ($action === 'approve') ? 'approved' : 'rejected';

$stmt = $pdo->prepare("
    UPDATE orders
    SET status = ?, admin_note = ?, admin_id = ?
    WHERE id = ?
");
$stmt->execute([
    $new_status,
    $admin_note,
    $_SESSION['admin_id'],
    $order_id
]);

/* =========================
   BUILD EMAIL
========================= */
$pickupInfo = '';
if (!empty($order['morning_pickup_time'])) {
    $pickupInfo = "<strong>Morning Pickup:</strong> "
        . date('g:i A', strtotime($order['morning_pickup_time'])) . "<br>";
}

if ($new_status === 'approved') {

$subject = "✅ Order Approved – {$order['order_code']}";

    $body = "
    <h3>Your order has been approved</h3>

    <p>
        <strong>Order ID:</strong> {$order['order_code']}<br>
        <strong>Food Date:</strong> {$order['order_for_date']}<br>
        {$pickupInfo}
        <strong>Total:</strong> RM " . number_format($order['total_amount'], 2) . "
    </p>

    <p>Your payment has been verified and your order is confirmed.</p>

    <p>Thank you,<br>" . SITE_NAME . "</p>
    ";

} else {

    $subject = "Order Rejected - {$order['order_code']}";

    $body = "
    <h3>Your order has been rejected</h3>

    <p>
        <strong>Order ID:</strong> {$order['order_code']}<br>
        <strong>Food Date:</strong> {$order['order_for_date']}
    </p>

    <p><strong>Reason:</strong></p>
    <p>" . nl2br(htmlspecialchars($admin_note)) . "</p>

    <p>Please contact the canteen admin for more information.</p>

    <p>Thank you,<br>" . SITE_NAME . "</p>
    ";
}

/* =========================
   SEND EMAIL (WITH CHECK)
========================= */
$result = send_email($order['employee_email'], $subject, $body);

if (!$result) {
    // Log for admin / server visibility
    error_log(
        "EMAIL FAILED | Order: {$order['order_code']} | To: {$order['employee_email']}"
    );

    // Optional: show warning during testing
    // die("Order updated, but email failed. Please check mail configuration.");
}

/* =========================
   REDIRECT BACK
========================= */
header("Location: view_orders.php?order_id=" . $order_id);
exit;
