<?php
session_start();
require_once __DIR__ . '/../includes/db.php';

// Ensure uploads folder exists
$uploadDir = __DIR__ . '/../uploads/';
if (!is_dir($uploadDir)) mkdir($uploadDir, 0755, true);

// Handle new menu submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_menu'])) {

    $title        = trim($_POST['title']);
    $description  = trim($_POST['description'] ?? '');
    $price        = floatval($_POST['price'] ?? 0);
    $meal_session = $_POST['meal_session'] ?? 'morning';
    $published    = isset($_POST['published']) ? 1 : 0;
    $imagePath    = null;

    // Handle image upload
    if (!empty($_FILES['image']['name']) && $_FILES['image']['error'] === 0) {
        $ext = pathinfo($_FILES['image']['name'], PATHINFO_EXTENSION);
        $newFileName = uniqid('menu_') . "." . $ext;
        move_uploaded_file($_FILES['image']['tmp_name'], $uploadDir . $newFileName);
        $imagePath = $newFileName;
    }

    // Validation
    if ($title === '') {
        $error = "Menu title is required.";
    } elseif ($price <= 0) {
        $error = "Price must be greater than 0.";
    }

    // Insert
    if (empty($error)) {
        $stmt = $pdo->prepare("
            INSERT INTO menu_items 
            (title, description, price, meal_session, published, image)
            VALUES (?, ?, ?, ?, ?, ?)
        ");

        if ($stmt->execute([
            $title,
            $description,
            $price,
            $meal_session,
            $published,
            $imagePath
        ])) {
            $success = "Menu item added successfully!";
        } else {
            $error = "Failed to add menu item.";
        }
    }
}

// Toggle publish
if (isset($_GET['toggle_publish'])) {
    $id = (int)$_GET['toggle_publish'];
    $pdo->prepare("
        UPDATE menu_items 
        SET published = IF(published=1,0,1)
        WHERE id = ?
    ")->execute([$id]);

    header("Location: add_menu.php");
    exit;
}

// Fetch menu
$menuItems = $pdo->query("SELECT * FROM menu_items ORDER BY id DESC")->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html>
<head>
<meta charset="UTF-8">
<title>Add Menu</title>
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
</head>

<body class="bg-light">
<div class="container py-5">

<h2>Add Menu Item</h2>
<a href="dashboard.php" class="btn btn-secondary mb-3">&larr; Back</a>

<?php if (!empty($error)) echo "<div class='alert alert-danger'>$error</div>"; ?>
<?php if (!empty($success)) echo "<div class='alert alert-success'>$success</div>"; ?>

<form method="post" enctype="multipart/form-data" class="mb-5">
<input type="hidden" name="add_menu" value="1">

<div class="mb-3">
<label class="form-label">Menu Title</label>
<input type="text" name="title" class="form-control" required>
</div>

<div class="mb-3">
<label class="form-label">Description</label>
<textarea name="description" class="form-control"></textarea>
</div>

<div class="mb-3">
<label class="form-label">Price (RM)</label>
<input type="number" name="price" step="0.01" class="form-control" required>
</div>

<div class="mb-3">
<label class="form-label">Meal Session</label>
<select name="meal_session" class="form-select" required>
    <option value="morning">Morning</option>
    <option value="afternoon">Afternoon</option>
</select>
</div>

<div class="mb-3">
<label class="form-label">Food Image</label>
<input type="file" name="image" class="form-control" accept="image/*">
</div>

<div class="form-check mb-3">
<input class="form-check-input" type="checkbox" name="published" value="1" id="publish">
<label class="form-check-label" for="publish">Publish now</label>
</div>

<button class="btn btn-primary">Add Menu</button>
</form>

<h3>Existing Menu</h3>
<table class="table table-bordered align-middle">
<thead class="table-dark">
<tr>
<th>ID</th>
<th>Image</th>
<th>Title</th>
<th>Session</th>
<th>Price</th>
<th>Status</th>
<th>Action</th>
</tr>
</thead>

<tbody>
<?php foreach ($menuItems as $item): ?>
<tr>
<td><?= $item['id'] ?></td>
<td>
<?php if ($item['image']): ?>
<img src="../uploads/<?= $item['image'] ?>" width="60">
<?php else: ?> -
<?php endif; ?>
</td>
<td><?= htmlspecialchars($item['title']) ?></td>
<td>
<span class="badge <?= $item['meal_session']=='morning'?'bg-info':'bg-warning text-dark' ?>">
<?= ucfirst($item['meal_session']) ?>
</span>
</td>
<td>RM <?= number_format($item['price'],2) ?></td>
<td><?= $item['published']?'Yes':'No' ?></td>
<td>
<a class="btn btn-sm btn-warning" href="?toggle_publish=<?= $item['id'] ?>">Publish</a>
<a class="btn btn-sm btn-info" href="edit_menu.php?id=<?= $item['id'] ?>">Edit</a>
</td>
</tr>
<?php endforeach; ?>
</tbody>
</table>

</div>
</body>
</html>

<?php include __DIR__ . '/../includes/footer.php'; ?>